import { useStore, Model3D } from '../store/useStore'
import { motion } from 'framer-motion'
import { 
  MapPin, 
  Calendar, 
  Navigation, 
  Box, 
  MoreVertical,
  Eye,
  Trash2,
  Download,
  Share2,
  Sparkles,
  Upload,
  FileBox
} from 'lucide-react'
import { useState, useCallback, useEffect } from 'react'

function ModelCard({ model, index }: { model: Model3D; index: number }) {
  const { selectModel, removeModel } = useStore()
  const [menuOpen, setMenuOpen] = useState(false)
  const [copied, setCopied] = useState(false)

  const formatNumber = (num: number) => {
    if (num >= 1000000) return (num / 1000000).toFixed(1) + 'M'
    if (num >= 1000) return (num / 1000).toFixed(1) + 'K'
    return num.toString()
  }

  const getFormatColor = (format: string) => {
    const colors: Record<string, string> = {
      glb: 'bg-emerald-100 text-emerald-700',
      gltf: 'bg-emerald-100 text-emerald-700',
      obj: 'bg-blue-100 text-blue-700',
      fbx: 'bg-purple-100 text-purple-700',
      ply: 'bg-amber-100 text-amber-700',
      las: 'bg-rose-100 text-rose-700',
      laz: 'bg-rose-100 text-rose-700',
      splat: 'bg-cyan-100 text-cyan-700'
    }
    return colors[format] || 'bg-gray-100 text-gray-700'
  }

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ delay: index * 0.1, duration: 0.4 }}
      className="model-card bg-white rounded-2xl shadow-soft overflow-hidden group"
    >
      {/* Thumbnail */}
      <div 
        className="relative h-48 bg-gradient-to-br from-surface-100 to-surface-200 overflow-hidden cursor-pointer"
        onClick={() => selectModel(model)}
      >
        {/* Placeholder 3D preview */}
        <div className="absolute inset-0 flex items-center justify-center">
          <div className="relative">
            <Box size={64} className="text-surface-300" strokeWidth={1} />
            <motion.div 
              className="absolute inset-0 flex items-center justify-center"
              animate={{ rotateY: 360 }}
              transition={{ duration: 8, repeat: Infinity, ease: "linear" }}
            >
              <div className="w-12 h-12 border-2 border-dashed border-primary-300 rounded-lg" />
            </motion.div>
          </div>
        </div>
        
        {/* Overlay on hover */}
        <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity flex items-end justify-center pb-4">
          <button className="flex items-center gap-2 px-4 py-2 bg-white/90 backdrop-blur rounded-lg font-medium text-sm hover:bg-white transition-colors">
            <Eye size={16} />
            View Model
          </button>
        </div>

        {/* Format badge */}
        <div className="absolute top-3 left-3">
          <span className={`px-2 py-1 rounded-md text-xs font-semibold uppercase ${getFormatColor(model.format)}`}>
            {model.format}
          </span>
        </div>

        {/* Menu button */}
        <div className="absolute top-3 right-3">
          <button 
            onClick={(e) => { e.stopPropagation(); setMenuOpen(!menuOpen) }}
            className="p-1.5 bg-white/80 backdrop-blur rounded-lg opacity-0 group-hover:opacity-100 transition-opacity hover:bg-white"
          >
            <MoreVertical size={16} />
          </button>
          
          {menuOpen && (
            <motion.div 
              initial={{ opacity: 0, scale: 0.95 }}
              animate={{ opacity: 1, scale: 1 }}
              className="absolute right-0 top-full mt-1 bg-white rounded-lg shadow-medium py-1 min-w-[140px] z-10"
            >
              {model.shareUrl && (
                <button 
                  onClick={(e) => { 
                    e.stopPropagation()
                    const fullUrl = `${window.location.origin}${model.shareUrl}`
                    navigator.clipboard.writeText(fullUrl)
                    setCopied(true)
                    setTimeout(() => setCopied(false), 2000)
                  }}
                  className="w-full px-3 py-2 text-left text-sm hover:bg-surface-50 flex items-center gap-2"
                >
                  <Share2 size={14} /> {copied ? 'Copied!' : 'Copy Link'}
                </button>
              )}
              <button className="w-full px-3 py-2 text-left text-sm hover:bg-surface-50 flex items-center gap-2">
                <Download size={14} /> Download
              </button>
              <hr className="my-1 border-surface-100" />
              <button 
                onClick={(e) => { e.stopPropagation(); removeModel(model.id) }}
                className="w-full px-3 py-2 text-left text-sm hover:bg-red-50 text-red-600 flex items-center gap-2"
              >
                <Trash2 size={14} /> Delete
              </button>
            </motion.div>
          )}
        </div>
      </div>

      {/* Info */}
      <div className="p-4">
        <h3 className="font-display font-semibold text-lg mb-1 line-clamp-1">{model.name}</h3>
        {model.description && (
          <p className="text-sm text-surface-500 mb-3 line-clamp-2">{model.description}</p>
        )}

        {/* Metadata */}
        <div className="flex flex-wrap gap-2 text-xs text-surface-500">
          {model.metadata?.location && (
            <span className="flex items-center gap-1 bg-surface-50 px-2 py-1 rounded">
              <MapPin size={12} />
              {model.metadata.location}
            </span>
          )}
          {model.metadata?.captureDate && (
            <span className="flex items-center gap-1 bg-surface-50 px-2 py-1 rounded">
              <Calendar size={12} />
              {new Date(model.metadata.captureDate).toLocaleDateString()}
            </span>
          )}
          {model.metadata?.droneModel && (
            <span className="flex items-center gap-1 bg-surface-50 px-2 py-1 rounded">
              <Navigation size={12} />
              {model.metadata.droneModel}
            </span>
          )}
        </div>

        {/* Stats */}
        {model.metadata?.polygons && (
          <div className="mt-3 pt-3 border-t border-surface-100 flex items-center justify-between text-sm">
            <span className="text-surface-500">
              {formatNumber(model.metadata.polygons)} polygons
            </span>
            <span className="text-surface-400">{model.metadata.fileSize}</span>
          </div>
        )}
      </div>
    </motion.div>
  )
}

function UploadZone() {
  const { uploadModel, isLoading } = useStore()
  const [isDragging, setIsDragging] = useState(false)
  const [uploadProgress, setUploadProgress] = useState<Record<string, number>>({})

  const supportedExtensions = ['glb', 'gltf', 'obj', 'fbx', 'ply', 'las', 'laz', 'splat']
  
  const processFile = useCallback(async (file: File) => {
    const ext = file.name.split('.').pop()?.toLowerCase() || ''
    
    // Check if it's a supported format
    if (!supportedExtensions.includes(ext)) {
      console.log(`Skipping unsupported file: ${file.name}`)
      return
    }
    
    try {
      setUploadProgress(prev => ({ ...prev, [file.name]: 0 }))
      await uploadModel(file, {
        name: file.name.replace(/\.[^/.]+$/, '')
      })
      setUploadProgress(prev => {
        const next = { ...prev }
        delete next[file.name]
        return next
      })
    } catch (error) {
      console.error('Failed to upload:', file.name, error)
      setUploadProgress(prev => {
        const next = { ...prev }
        delete next[file.name]
        return next
      })
    }
  }, [uploadModel])

  // Handle file entries from folder drops
  const processEntry = useCallback(async (entry: FileSystemEntry): Promise<void> => {
    if (entry.isFile) {
      const fileEntry = entry as FileSystemFileEntry
      return new Promise((resolve) => {
        fileEntry.file((file) => {
          processFile(file)
          resolve()
        })
      })
    } else if (entry.isDirectory) {
      const dirEntry = entry as FileSystemDirectoryEntry
      const reader = dirEntry.createReader()
      return new Promise((resolve) => {
        reader.readEntries(async (entries) => {
          for (const e of entries) {
            await processEntry(e)
          }
          resolve()
        })
      })
    }
  }, [processFile])

  const handleDrop = useCallback(async (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault()
    e.stopPropagation()
    setIsDragging(false)
    
    const items = e.dataTransfer.items
    console.log('Items dropped:', items.length)
    
    // Process all dropped items (files and folders)
    const entries: FileSystemEntry[] = []
    for (let i = 0; i < items.length; i++) {
      const entry = items[i].webkitGetAsEntry()
      if (entry) {
        entries.push(entry)
      }
    }
    
    // Process entries (handles both files and folders)
    for (const entry of entries) {
      await processEntry(entry)
    }
    
    // Fallback for browsers that don't support webkitGetAsEntry
    if (entries.length === 0) {
      const files = Array.from(e.dataTransfer.files)
      files.forEach(processFile)
    }
  }, [processEntry, processFile])

  const handleDragOver = useCallback((e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault()
    e.stopPropagation()
    setIsDragging(true)
  }, [])

  const handleDragLeave = useCallback((e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault()
    e.stopPropagation()
    setIsDragging(false)
  }, [])

  const handleClick = useCallback(() => {
    const input = document.createElement('input')
    input.type = 'file'
    input.multiple = true
    input.accept = supportedExtensions.map(ext => `.${ext}`).join(',')
    input.onchange = (e) => {
      const files = (e.target as HTMLInputElement).files
      if (files) {
        Array.from(files).forEach(processFile)
      }
    }
    input.click()
  }, [processFile])

  const isDragActive = isDragging

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ delay: 0.2 }}
      onDrop={handleDrop}
      onDragOver={handleDragOver}
      onDragLeave={handleDragLeave}
      onClick={handleClick}
      className={`drop-zone border-2 border-dashed rounded-2xl p-8 text-center cursor-pointer transition-all
        ${isDragActive ? 'active border-primary-400 bg-primary-50' : 'border-surface-300 hover:border-primary-300 hover:bg-surface-50'}`}
    >
      <div className="flex flex-col items-center gap-3">
        <div className={`w-16 h-16 rounded-2xl flex items-center justify-center transition-colors
          ${isDragActive ? 'bg-primary-100' : 'bg-surface-100'}`}>
          <Upload size={28} className={isDragActive ? 'text-primary-500' : 'text-surface-400'} />
        </div>
        <div>
          <p className="font-medium text-surface-700">
            {isDragActive ? 'Drop your 3D models here' : 'Drag & drop your 3D models or folders'}
          </p>
          <p className="text-sm text-surface-400 mt-1">
            Supports GLTF, GLB, OBJ, FBX, PLY, LAS/LAZ, SPLAT
          </p>
          <p className="text-xs text-surface-300 mt-2">
            Click to browse files
          </p>
          {isLoading && (
            <div className="mt-4">
              <div className="w-full bg-surface-200 rounded-full h-2">
                <div className="bg-primary-500 h-2 rounded-full animate-pulse" style={{ width: '60%' }} />
              </div>
              <p className="text-xs text-surface-500 mt-2">Uploading...</p>
            </div>
          )}
        </div>
      </div>
    </motion.div>
  )
}

export function Gallery() {
  const { models, isLoading, loadModels } = useStore()
  
  useEffect(() => {
    loadModels()
  }, [loadModels])

  return (
    <div className="max-w-[1800px] mx-auto px-4 lg:px-6 py-8">
      {/* Hero section */}
      <motion.div 
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className="text-center mb-12"
      >
        <div className="inline-flex items-center gap-2 px-3 py-1.5 bg-primary-50 rounded-full text-primary-600 text-sm font-medium mb-4">
          <Sparkles size={14} />
          <span>Drone 3D Model Showcase</span>
        </div>
        <h1 className="font-display text-4xl md:text-5xl font-bold mb-4">
          Your <span className="gradient-text">Aerial Captures</span>
        </h1>
        <p className="text-surface-500 max-w-2xl mx-auto text-lg">
          Explore and share your drone-captured 3D models. Upload photogrammetry outputs, 
          point clouds, and Gaussian splats from any software.
        </p>
      </motion.div>

      {/* Stats bar */}
      <motion.div 
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.1 }}
        className="flex flex-wrap items-center justify-center gap-6 mb-8 py-4"
      >
        <div className="flex items-center gap-2 text-surface-600">
          <FileBox size={20} className="text-primary-500" />
          <span className="font-semibold">{models.length}</span>
          <span className="text-surface-400">Models</span>
        </div>
        <div className="w-px h-6 bg-surface-200" />
        <div className="flex items-center gap-2">
          {['glb', 'obj', 'ply', 'splat'].map(format => (
            <span key={format} className="px-2 py-1 bg-surface-100 rounded text-xs font-medium text-surface-500 uppercase">
              {format}
            </span>
          ))}
        </div>
      </motion.div>

      {/* Upload zone */}
      <div className="mb-8">
        <UploadZone />
      </div>

      {/* Model grid */}
      {models.length > 0 ? (
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
          {models.map((model, index) => (
            <ModelCard key={model.id} model={model} index={index} />
          ))}
        </div>
      ) : (
        <div className="text-center py-16">
          <Box size={64} className="mx-auto text-surface-300 mb-4" strokeWidth={1} />
          <p className="text-surface-500">No models yet. Upload your first 3D model to get started!</p>
        </div>
      )}
    </div>
  )
}

